/**
 * Node-RED node: wait-i2c
 * Espera a que el bus I2C estigui lliure abans de deixar passar el missatge
 * 
 * Aquest node actua com un semàfor que evita col·lisions I2C:
 * - Si el bus està lliure, deixa passar el missatge immediatament
 * - Si el bus està ocupat, espera fins que estigui lliure
 * - Garanteix que no hi hagi operacions I2C simultànies
 */

module.exports = function(RED) {
    const { isBusOperating, waitForBusFree } = require('./i2c-iotv4');

    function WaitI2CNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        // Configuració del node
        const maxWaitTime = parseInt(config.maxWaitTime) || 1000; // Temps màxim d'espera en ms
        const showStatus = config.showStatus !== false; // Per defecte true

        node.on('input', async function(msg) {
            try {
                // Mostra estat d'espera
                if (showStatus) {
                    node.status({fill: "yellow", shape: "ring", text: "esperant bus..."});
                }
                
                // Espera a que el bus I2C estigui lliure
                const startTime = Date.now();
                await waitForBusFree(maxWaitTime);
                const waitTime = Date.now() - startTime;
                
                // Afegeix informació al missatge
                msg.i2cWaitTime = waitTime;
                
                // Mostra estat lliure
                if (showStatus) {
                    if (waitTime > 0) {
                        node.status({fill: "green", shape: "dot", text: `esperat ${waitTime}ms`});
                    } else {
                        node.status({fill: "green", shape: "dot", text: "bus lliure"});
                    }
                }
                
                // Envia el missatge
                node.send(msg);
                
                // Neteja l'estat després d'1 segon
                setTimeout(() => {
                    if (showStatus) {
                        node.status({});
                    }
                }, 1000);
                
            } catch (error) {
                node.error(`Error esperant bus I2C: ${error.message}`, msg);
                node.status({fill: "red", shape: "ring", text: "error"});
                
                // Opcional: envia el missatge igualment (comentat per seguretat)
                // node.send(msg);
            }
        });
        
        // Neteja l'estat quan el node es tanca
        node.on('close', function() {
            node.status({});
        });
    }

    RED.nodes.registerType("wait-i2c", WaitI2CNode);
}
