/**
 * Test de Validació v1.1.14
 * 
 * Valida que el paquet optimitzat funciona correctament
 * Executa aquest test després d'instal·lar la versió 1.1.14
 */

const { din, dout, openBus, closeBus } = require('./package/i2c-iotv4.js');

const DIN_ADDR = "0000";
const DIN_SIDE = "A";
const DOUT_ADDR = "0000";
const DOUT_SIDE = "B";
const DOUBLE_READING = true;

let qMemory = 0x00;

console.log('═'.repeat(80));
console.log('✅ TEST DE VALIDACIÓ - node-red-contrib-i2c-iotv v1.1.14');
console.log('═'.repeat(80));
console.log('');

async function runValidationTest() {
    const cycleTimes = [];
    let errors = 0;
    const numCycles = 100;
    
    console.log(`🧪 Executant ${numCycles} cicles de validació...`);
    console.log('');
    
    for (let i = 0; i < numCycles; i++) {
        const cycleStart = Date.now();
        
        try {
            // Executa din i dout en paral·lel (com fa Node-RED)
            const [binaryResult] = await Promise.all([
                din(DIN_ADDR, DIN_SIDE, DOUBLE_READING),
                dout(DOUT_ADDR, DOUT_SIDE, qMemory)
            ]);
            
            const decimalValue = parseInt(binaryResult, 2);
            const cycleTime = Date.now() - cycleStart;
            cycleTimes.push(cycleTime);
            
            if (decimalValue === 65) {
                errors++;
            }
            
            // Progrés cada 20 cicles
            if ((i + 1) % 20 === 0) {
                process.stdout.write(`   ${i + 1}/${numCycles} cicles - Errors: ${errors}     \r`);
            }
            
        } catch (error) {
            errors++;
            console.error(`\n   ❌ Error en cicle ${i + 1}: ${error.message}`);
        }
    }
    
    console.log(''); // Nova línia
    console.log('');
    
    // Resultats
    const avgCycleTime = cycleTimes.reduce((a, b) => a + b, 0) / cycleTimes.length;
    const minCycleTime = Math.min(...cycleTimes);
    const maxCycleTime = Math.max(...cycleTimes);
    const errorPct = (errors / numCycles) * 100;
    
    console.log('─'.repeat(80));
    console.log('📊 RESULTATS DE LA VALIDACIÓ');
    console.log('─'.repeat(80));
    console.log(`   Cicles executats: ${numCycles}`);
    console.log(`   Errors: ${errors} (${errorPct.toFixed(2)}%)`);
    console.log('');
    console.log(`   Temps de cicle:`);
    console.log(`     - Mitjà: ${avgCycleTime.toFixed(1)}ms`);
    console.log(`     - Mínim: ${minCycleTime.toFixed(1)}ms`);
    console.log(`     - Màxim: ${maxCycleTime.toFixed(1)}ms`);
    console.log('');
    
    // Avaluació
    let status = '✅ PERFECTE';
    let recommendation = '';
    
    if (errors > 0) {
        status = '❌ AMB ERRORS';
        recommendation = '\n   ⚠️  Hi ha errors! Verifica:\n' +
                        '      - Connexions I2C\n' +
                        '      - Alimentació dels dispositius\n' +
                        '      - Que no hi hagi altres processos usant I2C';
    } else if (avgCycleTime > 40) {
        status = '✅ CORRECTE';
        recommendation = `\n   💡 Recomanació: Pots usar cicles de 50ms amb seguretat\n` +
                        `      Temps mitjà (${avgCycleTime.toFixed(1)}ms) + marge = ~50ms òptim`;
    } else if (avgCycleTime <= 40) {
        status = '🚀 EXCEL·LENT';
        recommendation = `\n   🎉 Rendiment excel·lent!\n` +
                        `      Pots usar cicles de 40-50ms amb seguretat`;
    }
    
    console.log(`   Status: ${status}`);
    
    if (recommendation) {
        console.log(recommendation);
    }
    
    console.log('');
    console.log('─'.repeat(80));
    console.log('');
    
    // Validació vs especificacions
    const isValid = errors === 0 && avgCycleTime <= 42;
    
    if (isValid) {
        console.log('✅ VALIDACIÓ SUPERADA');
        console.log('   El paquet v1.1.14 funciona correctament');
        console.log('   Rendiment dins de les especificacions');
        console.log('');
        console.log('📋 RESUM:');
        console.log(`   - Millora vs v1.1.13: ~${(42 - avgCycleTime).toFixed(1)}ms (${((42 - avgCycleTime) / 42 * 100).toFixed(1)}%)`);
        console.log(`   - Cicle recomanat per Node-RED: 50ms`);
        console.log(`   - Cicle mínim viable: 40ms (avançat)`);
    } else {
        console.log('⚠️  VALIDACIÓ FALLIDA');
        if (errors > 0) {
            console.log(`   Motiu: ${errors} errors detectats`);
        }
        if (avgCycleTime > 42) {
            console.log(`   Motiu: Cicle massa lent (${avgCycleTime.toFixed(1)}ms > 42ms esperat)`);
        }
        console.log('');
        console.log('   Accions recomanades:');
        console.log('   1. Verifica les connexions I2C');
        console.log('   2. Comprova que cap altre procés usa /dev/i2c-1');
        console.log('   3. Reinicia Node-RED si estava executant-se');
        console.log('   4. Executa: sudo i2cdetect -y 1');
    }
    
    console.log('');
    console.log('═'.repeat(80));
    
    return isValid;
}

async function main() {
    try {
        console.log('📡 Obrint bus I2C...');
        await openBus();
        console.log('✅ Bus I2C obert');
        console.log('');
        
        const isValid = await runValidationTest();
        
        await closeBus();
        console.log('🔒 Bus I2C tancat');
        console.log('');
        
        process.exit(isValid ? 0 : 1);
        
    } catch (error) {
        console.error('❌ Error fatal:', error);
        console.error(error.stack);
        await closeBus();
        process.exit(1);
    }
}

// Gestió de ctrl+C
process.on('SIGINT', async () => {
    console.log('\n\n⚠️  Test interromput');
    await closeBus();
    process.exit(0);
});

main();
