/**
 * Node-RED node: setiplc
 * Escriu les lectures digitals a variables de memòria I
 */

module.exports = function(RED) {
    // Funció local per inicialitzar variables I (sense dependències externes)
    function initVariableI(node, varname) {
        const store = node.context().global;
        const N_INDEX = 64;
        const ctxvar = Array(N_INDEX).fill(false);
        store.set(varname, ctxvar);
        return ctxvar;
    }

    function SetIPlcNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        node.iserror = false;

        node.on('input', function(msg) {
            try {
                // Validació del payload
                if (!msg.payload || typeof msg.payload !== 'object') {
                    node.error('El payload ha de ser un objecte JSON');
                    node.status({fill: "red", shape: "dot", text: "error: payload invàlid"});
                    return;
                }

                const { addr, side, binary, decimal } = msg.payload;

                // Validació dels camps requerits
                if (!addr || !side || !binary) {
                    node.error('Falten paràmetres: necessiten addr, side i binary');
                    node.status({fill: "red", shape: "dot", text: "error: paràmetres incomplets"});
                    return;
                }

                // Converteix l'adreça binària a decimal
                const addrDecimal = parseInt(addr, 2);
                if (isNaN(addrDecimal) || addrDecimal < 0 || addrDecimal > 15) {
                    node.error('Adreça invàlida: ' + addr);
                    node.status({fill: "red", shape: "dot", text: "error: adreça invàlida"});
                    return;
                }

                // Determina el nom de la variable I
                const varname = "I" + addrDecimal;
                
                // Obté la variable del context global
                const store = node.context().global;
                let ctxvar = store.get(varname);

                // Si la variable no existeix, crea-la
                if (ctxvar === undefined) {
                    node.warn(`Variable ${varname} no existeix, creant-la...`);
                    ctxvar = initVariableI(node, varname);
                    
                    if (ctxvar === undefined) {
                        node.error(`No s'ha pogut crear la variable ${varname}`);
                        node.status({fill: "red", shape: "dot", text: "error: no es pot crear I"});
                        return;
                    }
                }

                // Validació de la cadena binària
                if (binary.length !== 8) {
                    node.error('La cadena binària ha de tenir 8 bits');
                    node.status({fill: "red", shape: "dot", text: "error: binary invàlid"});
                    return;
                }

                // Determina l'offset segons el costat (A=0, B=10)
                const sideLower = side.toLowerCase();
                let offset = 0;
                
                if (sideLower === 'a') {
                    offset = 0;
                } else if (sideLower === 'b') {
                    offset = 10;
                } else {
                    node.error('Side ha de ser A o B');
                    node.status({fill: "red", shape: "dot", text: "error: side invàlid"});
                    return;
                }

                // Escriu cada bit a la variable I
                // La cadena binària està ordenada com: bit7 bit6 bit5 bit4 bit3 bit2 bit1 bit0
                for (let bitPos = 0; bitPos < 8; bitPos++) {
                    const stringIndex = binary.length - 1 - bitPos; // Índex dins la cadena
                    const bitChar = binary[stringIndex];
                    const bitValue = bitChar === '1' ? true : false;
                    
                    const memIndex = offset + bitPos;
                    
                    // Assegura't que no sortim dels límits de l'array (64 elements)
                    if (memIndex < 64) {
                        ctxvar[memIndex] = bitValue;
                    }
                }

                // Actualitza la variable al context global
                store.set(varname, ctxvar);

                // Mostra l'estat
                const statusText = `${varname}.${offset}-${offset+7}: ${binary} (${decimal})`;
                node.status({fill: "green", shape: "dot", text: statusText});

                // Reenvia el missatge
                node.send(msg);

            } catch (error) {
                node.error(`Error en setiplc: ${error.message}`);
                node.status({fill: "red", shape: "ring", text: "error"});
            }
        });
    }

    RED.nodes.registerType("setiplc", SetIPlcNode);
}
