/**
 * Node-RED node: setdti
 * Escriu valors a les variables DTI des de missatges MQTT
 * Format del topic: /<MAC>/I<address>/<index>
 * Exemple: /B827EB123456/I0/6 amb payload true → DTI0[6] = true
 */

module.exports = function(RED) {
    
    // Funció per inicialitzar variables DTI si no existeixen
    function initVariableDTI(node, varname) {
        const store = node.context().global;
        const N_INDEX = 64;
        const ctxvar = Array(N_INDEX).fill(false);
        store.set(varname, ctxvar);
        return ctxvar;
    }
    
    // Funció per convertir el payload a boolean
    function parseBoolean(payload) {
        if (typeof payload === 'boolean') {
            return payload;
        }
        if (typeof payload === 'number') {
            return payload !== 0;
        }
        if (typeof payload === 'string') {
            const lower = payload.toLowerCase().trim();
            if (lower === 'true' || lower === '1') return true;
            if (lower === 'false' || lower === '0') return false;
        }
        return null; // Valor no vàlid
    }

    function SetDtiNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        node.iserror = false;

        node.on('input', function(msg) {
            try {
                // Validació del topic
                if (!msg.topic || typeof msg.topic !== 'string') {
                    node.error('El topic ha de ser un string vàlid');
                    node.status({fill: "red", shape: "dot", text: "error: topic invàlid"});
                    return;
                }

                // Parse del topic: /<MAC>/I<address>/<index>
                // Exemple: /B827EB123456/I0/6
                const topicPattern = /^\/([0-9A-F]{12})\/I(\d+)\/(\d+)$/;
                const match = msg.topic.match(topicPattern);

                if (!match) {
                    node.error(`Format de topic invàlid: ${msg.topic}. Esperat: /<MAC>/I<address>/<index>`);
                    node.status({fill: "red", shape: "dot", text: "error: format topic"});
                    return;
                }

                const mac = match[1];           // MAC (ex: B827EB123456)
                const address = parseInt(match[2], 10);  // Address (ex: 0)
                const index = parseInt(match[3], 10);    // Index (ex: 6)

                // Validació de l'address
                if (isNaN(address) || address < 0 || address > 999) {
                    node.error(`Address invàlid: ${address}. Ha d'estar entre 0 i 999`);
                    node.status({fill: "red", shape: "dot", text: "error: address invàlid"});
                    return;
                }

                // Validació de l'index
                if (isNaN(index) || index < 0 || index > 63) {
                    node.error(`Index invàlid: ${index}. Ha d'estar entre 0 i 63`);
                    node.status({fill: "red", shape: "dot", text: "error: index invàlid"});
                    return;
                }

                // Convertir el payload a boolean
                const value = parseBoolean(msg.payload);
                if (value === null) {
                    node.error(`Payload invàlid: ${msg.payload}. Ha de ser true/false, 1/0`);
                    node.status({fill: "red", shape: "dot", text: "error: payload invàlid"});
                    return;
                }

                // Determina el nom de la variable DTI
                const varname = "DTI" + address;
                
                // Obté la variable del context global
                const store = node.context().global;
                let ctxvar = store.get(varname);

                // Si la variable no existeix, crea-la
                if (ctxvar === undefined) {
                    node.warn(`Variable ${varname} no existeix, creant-la...`);
                    ctxvar = initVariableDTI(node, varname);
                    
                    if (ctxvar === undefined) {
                        node.error(`No s'ha pogut crear la variable ${varname}`);
                        node.status({fill: "red", shape: "dot", text: "error: no es pot crear DTI"});
                        return;
                    }
                }

                // Escriu el valor a la variable DTI
                ctxvar[index] = value;

                // Actualitza la variable al context global
                store.set(varname, ctxvar);

                // Prepara la sortida
                msg.payload = {
                    mac: mac,
                    variable: varname,
                    index: index,
                    value: value,
                    success: true,
                    originalTopic: msg.topic
                };

                // Mostra l'estat
                const statusText = `${varname}[${index}] = ${value}`;
                node.status({fill: "green", shape: "dot", text: statusText});

                // Reenvia el missatge
                node.send(msg);

            } catch (error) {
                node.error(`Error en setdti: ${error.message}`);
                node.status({fill: "red", shape: "ring", text: "error"});
                msg.payload = { 
                    error: error.message,
                    originalTopic: msg.topic,
                    success: false
                };
                node.send(msg);
            }
        });
    }

    RED.nodes.registerType("setdti", SetDtiNode);
}
