/**
 * Node-RED node: qchanges
 * Detecta i reporta els canvis en les variables Q (Output)
 * Retorna un array amb els grups de 8 bits que han canviat
 */

module.exports = function(RED) {
    
    function QChangesNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        // Emmagatzematge de l'estat anterior de les variables Q
        // Format: { "Q0_A": 0, "Q0_B": 0, "Q1_A": 0, ... }
        node.previousState = {};

        // Funció per convertir un número decimal a binari de 4 bits
        function decimalToBinary4(num) {
            return num.toString(2).padStart(4, '0');
        }

        // Funció per extreure els 8 bits d'una variable Q
        function extractByte(qVar, offset) {
            let value = 0;
            for (let bitPos = 0; bitPos < 8; bitPos++) {
                const memIndex = offset + bitPos;
                if (memIndex < 64 && qVar[memIndex] === true) {
                    value |= (1 << bitPos);
                }
            }
            return value;
        }

        node.on('input', function(msg) {
            try {
                node.status({fill: "blue", shape: "dot", text: "comprovant canvis..."});

                const store = node.context().global;
                const changes = [];

                // Busca totes les variables Q de 0 a 999
                for (let address = 0; address <= 999; address++) {
                    const varname = "Q" + address;
                    const qVar = store.get(varname);
                    
                    if (qVar !== undefined && Array.isArray(qVar)) {
                        // Converteix l'adreça a binari de 4 bits
                        const addrBinary = decimalToBinary4(address);
                        
                        // Comprova costat A (bits 0-7)
                        const sideAValue = extractByte(qVar, 0);
                        const sideAKey = varname + "_A";
                        const sideAPrevious = node.previousState[sideAKey];
                        
                        if (sideAPrevious === undefined || sideAPrevious !== sideAValue) {
                            // Hi ha hagut un canvi al costat A
                            const binaryStr = sideAValue.toString(2).padStart(8, '0');
                            changes.push({
                                addr: addrBinary,
                                side: "A",
                                binary: binaryStr,
                                decimal: sideAValue
                            });
                            // Actualitza l'estat anterior
                            node.previousState[sideAKey] = sideAValue;
                        }
                        
                        // Comprova costat B (bits 10-17)
                        const sideBValue = extractByte(qVar, 10);
                        const sideBKey = varname + "_B";
                        const sideBPrevious = node.previousState[sideBKey];
                        
                        if (sideBPrevious === undefined || sideBPrevious !== sideBValue) {
                            // Hi ha hagut un canvi al costat B
                            const binaryStr = sideBValue.toString(2).padStart(8, '0');
                            changes.push({
                                addr: addrBinary,
                                side: "B",
                                binary: binaryStr,
                                decimal: sideBValue
                            });
                            // Actualitza l'estat anterior
                            node.previousState[sideBKey] = sideBValue;
                        }
                    }
                }

                // Mostra l'estat i envia missatge només si hi ha canvis
                if (changes.length === 0) {
                    node.status({fill: "green", shape: "dot", text: "cap canvi"});
                    // No enviar res si no hi ha canvis
                    return;
                } else {
                    const statusText = `${changes.length} canvi${changes.length !== 1 ? 's' : ''}`;
                    node.status({fill: "green", shape: "dot", text: statusText});
                    
                    // Prepara la sortida
                    msg.payload = changes;
                    
                    // Reenvia el missatge només si hi ha canvis
                    node.send(msg);
                }

            } catch (error) {
                node.error(`Error en qchanges: ${error.message}`);
                node.status({fill: "red", shape: "ring", text: "error"});
                msg.payload = [];
                node.send(msg);
            }
        });
    }

    RED.nodes.registerType("qchanges", QChangesNode);
}
