/**
 * Node-RED node: openclosei2c
 * Obre o tanca el bus I2C segons el valor booleà rebut
 */

module.exports = function(RED) {
    const iotv4 = require('./i2c-iotv4');

    function OpenCloseI2cNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;

        node.on('input', async function(msg) {
            try {
                // Validació del payload
                if (typeof msg.payload !== 'boolean') {
                    node.error('El payload ha de ser un booleà (true per obrir, false per tancar)');
                    node.status({fill: "red", shape: "dot", text: "error: payload invàlid"});
                    msg.payload = {
                        success: false,
                        error: 'El payload ha de ser un booleà'
                    };
                    node.send(msg);
                    return;
                }

                const shouldOpen = msg.payload;

                // OPEN BUS (true)
                if (shouldOpen) {
                    // Comprova si el bus ja està obert
                    if (iotv4.isBusOpen()) {
                        node.warn('El bus I2C ja està obert');
                        node.status({fill: "yellow", shape: "dot", text: "ja està obert"});
                        msg.payload = {
                            success: false,
                            action: 'open',
                            message: 'El bus I2C ja està obert',
                            alreadyOpen: true,
                            timestamp: new Date().toISOString()
                        };
                        node.send(msg);
                        return;
                    }

                    // Obre el bus I2C
                    await iotv4.openBus();
                    
                    node.status({fill: "green", shape: "dot", text: "Bus obert"});
                    
                    msg.payload = {
                        success: true,
                        action: 'open',
                        message: 'Bus I2C obert correctament',
                        timestamp: new Date().toISOString()
                    };
                    
                    node.send(msg);
                }
                // CLOSE BUS (false)
                else {
                    // Comprova si el bus està obert
                    if (!iotv4.isBusOpen()) {
                        node.warn('El bus I2C no està obert');
                        node.status({fill: "yellow", shape: "ring", text: "ja està tancat"});
                        msg.payload = {
                            success: false,
                            action: 'close',
                            message: 'El bus I2C no està obert',
                            alreadyClosed: true,
                            timestamp: new Date().toISOString()
                        };
                        node.send(msg);
                        return;
                    }

                    // Tanca el bus I2C
                    await iotv4.closeBus();
                    
                    node.status({fill: "grey", shape: "ring", text: "Bus tancat"});
                    
                    msg.payload = {
                        success: true,
                        action: 'close',
                        message: 'Bus I2C tancat correctament',
                        timestamp: new Date().toISOString()
                    };
                    
                    node.send(msg);
                }
                
            } catch (error) {
                node.status({fill: "red", shape: "ring", text: "Error"});
                node.error(`Error gestionant el bus I2C: ${error.message}`, msg);
                
                msg.payload = {
                    success: false,
                    action: msg.payload ? 'open' : 'close',
                    error: error.message,
                    timestamp: new Date().toISOString()
                };
                
                node.send(msg);
            }
        });

        node.on('close', function(done) {
            node.status({});
            // Intenta tancar el bus si està obert quan es tanca el node
            if (iotv4.isBusOpen()) {
                iotv4.closeBus()
                    .then(() => done())
                    .catch(() => done());
            } else {
                done();
            }
        });
    }

    RED.nodes.registerType("openclosei2c", OpenCloseI2cNode);
}

