/**
 * Node-RED node: iotv-q-memory
 * Crea i gestiona variables de memòria Q (Digital Output)
 * Compatible amb redplc-contact, getqplc, dout i doutbit
 */

module.exports = function(RED) {
    
    function IotvQMemoryNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;

        node.varname = "Q" + config.address;
        node.name = node.varname;
        node.persistent = config.persistent > 0;
        node.tupdate = config.tupdate;
        node.onwork = false;
        node.iserror = false;
        
        const N_INDEX = 64;
        
        // Inicialitza la variable amb els valors configurats
        const meminit = config.meminit || [];
        const ctxvar = Array(N_INDEX).fill(false);
        
        // Aplica els valors inicials configurats
        for (let i = 0; i < meminit.length && i < N_INDEX; i++) {
            ctxvar[i] = meminit[i] ? true : false;
        }
        
        // Guarda la variable al context global
        const store = node.context().global;
        store.set(node.varname, ctxvar);
        
        // Mostra l'estat
        const statustxt = node.persistent ? node.name + " ↻ " + node.tupdate + "s" : node.name;
        node.status({fill: "green", shape: "dot", text: statustxt});
        
        // Si persistent està habilitat, implementa la lògica de persistència
        if (node.persistent) {
            // Funció per actualitzar la persistència
            function update() {
                const currentVar = store.get(node.varname);
                const jsonCurrent = JSON.stringify(currentVar);
                
                // Aquí podries afegir lògica per guardar a fitxer
                // Per ara només actualitzem l'estat
                node.status({fill: "blue", shape: "dot", text: statustxt});
                
                setTimeout(() => {
                    node.status({fill: "green", shape: "dot", text: statustxt});
                    node.onwork = false;
                }, 100);
            }
            
            // Cicle d'actualització periòdic
            node.id_tupdate = setInterval(function() {
                if (!node.onwork) {
                    node.onwork = true;
                    update();
                }
            }, node.tupdate * 1000);
        }
        
        node.on('close', (removed, done) => {
            clearInterval(node.id_tupdate);
            
            if (node.persistent) {
                setTimeout(function() {
                    if (removed) {
                        // Si el node s'elimina, esborra la variable
                        store.set(node.varname, undefined);
                    }
                    done();
                }, 50);
            } else {
                // Si no és persistent, esborra la variable en tancar
                // Comentat per mantenir-la disponible per altres nodes
                // store.set(node.varname, undefined);
                done();
            }
        });
    }

    RED.nodes.registerType("iotv-q-memory", IotvQMemoryNode);
}
