/**
 * i2c-iotv4.js
 * Versió optimitzada amb gestió persistent del bus I2C
 * 
 * Millores respecte a v3:
 * - El bus I2C s'obre una sola vegada amb openBus()
 * - Totes les funcions utilitzen el bus obert
 * - El bus es tanca explícitament amb closeBus()
 * - Evita problemes de lectures incorrectes per obertures/tancaments continus
 * 
 * Es requereix el paquet 'i2c-bus'.
 */

const i2c = require('i2c-bus');
const I2C_BUS_NUMBER = 1;

// Variable global per mantenir el bus obert
let i2cBus = null;

let operationQueue = Promise.resolve();
let isOperating = false;

/**
 * Encua una operació I2C per evitar col·lisions
 */
function queueOperation(operation) {
    operationQueue = operationQueue
        .then(() => {
            isOperating = true;
            return operation();
        })
        .finally(() => {
            isOperating = false;
        });
    return operationQueue;
}

// --- Taules de Cerca d'Adreces I2C ---

const ANALOG_I2C_ADDRESSES = {
    '0000': 0x10, '0001': 0x11, '0010': 0x12, '0011': 0x13,
    '0100': 0x14, '0101': 0x15, '0110': 0x16, '0111': 0x17,
    '1000': 0x18, '1001': 0x19, '1010': 0x1A, '1011': 0x1B,
    '1100': 0x1C, '1101': 0x1D, '1110': 0x1E, '1111': 0x1F,
};

const DIGITAL_I2C_ADDRESSES = {
    '0000': 0x20, '0001': 0x21, '0010': 0x22, '0011': 0x23,
    '0100': 0x24, '0101': 0x25, '0110': 0x26, '0111': 0x27,
    '1000': 0x28, '1001': 0x29, '1010': 0x2A, '1011': 0x2B,
    '1100': 0x2C, '1101': 0x2D, '1110': 0x2E, '1111': 0x2F,
};

function getI2cAddr(addr, type) {
    if (type === 'analog') {
        return ANALOG_I2C_ADDRESSES[addr];
    } else if (type === 'digital') {
        return DIGITAL_I2C_ADDRESSES[addr];
    }
    return null;
}

// --- Funcions de Gestió del Bus I2C ---

/**
 * Obre el bus I2C i el manté obert per a futures operacions.
 * @returns {Promise<void>} Promesa que resol quan el bus s'ha obert correctament.
 */
function openBus() {
    return new Promise((resolve, reject) => {
        if (i2cBus !== null) {
            return reject(new Error('El bus I2C ja està obert. Tanca\'l abans d\'obrir-lo de nou.'));
        }
        
        i2cBus = i2c.open(I2C_BUS_NUMBER, (err) => {
            if (err) {
                i2cBus = null;
                return reject(err);
            }
            resolve();
        });
    });
}

/**
 * Tanca el bus I2C.
 * @returns {Promise<void>} Promesa que resol quan el bus s'ha tancat correctament.
 */
function closeBus() {
    return new Promise((resolve, reject) => {
        if (i2cBus === null) {
            return reject(new Error('El bus I2C no està obert.'));
        }
        
        i2cBus.close((err) => {
            if (err) {
                return reject(err);
            }
            i2cBus = null;
            resolve();
        });
    });
}

/**
 * Comprova si el bus I2C està obert.
 * @returns {boolean} True si el bus està obert, false altrament.
 */
function isBusOpen() {
    return i2cBus !== null;
}

// --- Funcions Utilitàries per a Lectura/Escriptura I2C ---

/**
 * Funció auxiliar per realitzar doble lectura de Word.
 * @param {number} i2cAddr Adreça I2C del dispositiu.
 * @param {number} command Registre/Comanda a llegir.
 * @returns {Promise<number>} Promesa que resol el valor de la segona lectura (Word de 16 bits).
 */
async function readWordDouble(i2cAddr, command) {
    if (!isBusOpen()) {
        throw new Error('El bus I2C no està obert. Crida openBus() primer.');
    }

    // Primera lectura (es descarta)
    await new Promise((resolve, reject) => {
        i2cBus.readWord(i2cAddr, command, (err, res1) => {
            if (err) return reject(err);
            resolve(res1);
        });
    });
    
    // Delay mínim entre lectures (optimitzat, igual que din)
    await new Promise(resolve => setTimeout(resolve, 1));
    
    // Segona lectura (el resultat vàlid)
    const res2 = await new Promise((resolve, reject) => {
        i2cBus.readWord(i2cAddr, command, (err, res2) => {
            if (err) return reject(err);
            resolve(res2);
        });
    });

    return res2;
}

/**
 * Funció auxiliar per realitzar doble lectura de Byte.
 * @param {number} i2cAddr Adreça I2C del dispositiu.
 * @param {number} command Registre/Comanda a llegir.
 * @returns {Promise<number>} Promesa que resol el valor de la segona lectura (Byte de 8 bits).
 */
async function readByteDouble(i2cAddr, command) {
    if (!isBusOpen()) {
        throw new Error('El bus I2C no està obert. Crida openBus() primer.');
    }

    // Primera lectura (es descarta)
    await new Promise((resolve, reject) => {
        i2cBus.readByte(i2cAddr, command, (err, res1) => {
            if (err) return reject(err);
            resolve(res1);
        });
    });

    // Delay mínim entre lectures (optimitzat de 30ms a 1ms)
    await new Promise(resolve => setTimeout(resolve, 1));
    
    // Segona lectura (el resultat vàlid)
    const res2 = await new Promise((resolve, reject) => {
        i2cBus.readByte(i2cAddr, command, (err, res2) => {
            if (err) return reject(err);
            resolve(res2);
        });
    });

    return res2;
}

/**
 * Funció auxiliar per a escriure un bloc I2C.
 * @param {number} i2cAddr Adreça I2C del dispositiu.
 * @param {number} command Registre/Comanda a escriure.
 * @param {Buffer} buffer Dades a escriure.
 * @returns {Promise<void>} Promesa que resol a l'èxit.
 */
async function writeI2cBlock(i2cAddr, command, buffer) {
    if (!isBusOpen()) {
        throw new Error('El bus I2C no està obert. Crida openBus() primer.');
    }

    await new Promise((resolve, reject) => {
        i2cBus.writeI2cBlock(i2cAddr, command, buffer.length, buffer, (err) => {
            if (err) return reject(err);
            resolve();
        });
    });
}

// --- Funcions del Mòdul ---

/**
 * Llegeix una entrada analògica (ADC).
 * @param {string} addr Adreça binària (e.g., '0000').
 * @param {string} side Costat ('a' o 'b').
 * @param {number} ndac Número del DAC (1-4).
 * @returns {Promise<number>} Promesa que resol al valor raw ADC (16 bits).
 */
async function ain(addr, side, ndac) {
    const i2cAddr = getI2cAddr(addr, 'analog');
    if (!i2cAddr) {
        throw new Error('Adreça analògica no vàlida.');
    }

    let addrChannel;
    const sideLower = side.toLowerCase();

    if (ndac === 1) {
        addrChannel = sideLower === 'a' ? 0x01 : 0x11;
    } else if (ndac === 2) {
        addrChannel = sideLower === 'a' ? 0x21 : 0x31;
    } else if (ndac === 3) {
        addrChannel = sideLower === 'a' ? 0x41 : 0x51;
    } else if (ndac === 4) {
        addrChannel = sideLower === 'a' ? 0x61 : 0x71;
    } else {
        throw new Error('ndac analògic no vàlid.');
    }

    const res = await readWordDouble(i2cAddr, addrChannel);
    return res;
}

/**
 * Llegeix una entrada analògica (ADC) i retorna el voltatge calculat.
 * @param {string} addr Adreça binària (e.g., '0000').
 * @param {string} side Costat ('a' o 'b').
 * @param {number} ndac Número del DAC (1-4).
 * @returns {Promise<object>} Promesa que resol a { raw: number, voltage: number, log: string }.
 */
async function ainv(addr, side, ndac) {
    const i2cAddr = getI2cAddr(addr, 'analog');
    if (!i2cAddr) {
        throw new Error('Adreça analògica no vàlida.');
    }

    let addrChannel;
    const sideLower = side.toLowerCase();

    if (ndac === 1) {
        addrChannel = sideLower === 'a' ? 0x01 : 0x11;
    } else if (ndac === 2) {
        addrChannel = sideLower === 'a' ? 0x21 : 0x31;
    } else if (ndac === 3) {
        addrChannel = sideLower === 'a' ? 0x41 : 0x51;
    } else if (ndac === 4) {
        addrChannel = sideLower === 'a' ? 0x61 : 0x71;
    } else {
        throw new Error('ndac analògic no vàlid.');
    }

    const res = await readWordDouble(i2cAddr, addrChannel);
    
    const voltage = (20 * parseFloat(res) / 26624) - 10;
    
    const logStr = `0x${res.toString(16).toUpperCase()} --> ${res} --> ${voltage.toFixed(2)}V`;
    console.log(`res: 0x${res.toString(16).toUpperCase()}: ${res}`);
    console.log(logStr);

    return { raw: res, voltage: voltage, log: logStr };
}

/**
 * Escriu una sortida analògica (DAC).
 * @param {string} addr Adreça binària (e.g., '0000').
 * @param {string} side Costat ('a' o 'b').
 * @param {number} ndac Número del DAC (1-4).
 * @param {number} value Valor (0-4095).
 * @returns {Promise<void>} Promesa que resol a l'èxit.
 */
/**
 * Converteix un voltatge (0-10V) a valor DAC (0-4095).
 * @param {number} voltage0_10 - Voltatge entre 0 i 10 volts.
 * @returns {number} - Valor DAC entre 0 i 4095.
 */
function v2aout(voltage0_10) {
    let retV = Math.round((voltage0_10 * 4095) / 10);
    if (retV < 0) {
        retV = 0;
    }
    if (retV > 4095) {
        retV = 4095;
    }
    return retV;
}

/**
 * Converteix un valor ADC (16 bits) a voltatge (-10V a +10V).
 * @param {number} ainValue - Valor ADC de 16 bits (0-65535).
 * @returns {number} - Voltatge entre -10V i +10V (arrodonit a 2 decimals).
 */
function ain2v(ainValue) {
    return Math.round((((20 * parseFloat(ainValue)) / 26624) - 10) * 100) / 100;
}

async function aout(addr, side, ndac, value) {
    const i2cAddr = getI2cAddr(addr, 'analog');
    if (!i2cAddr) {
        throw new Error('Adreça analògica no vàlida.');
    }

    let dacCommand;
    const sideLower = side.toLowerCase();

    if (ndac === 1) {
        dacCommand = sideLower === 'a' ? 0x02 : 0x12;
    } else if (ndac === 2) {
        dacCommand = sideLower === 'a' ? 0x22 : 0x32;
    } else if (ndac === 3) {
        dacCommand = sideLower === 'a' ? 0x42 : 0x52;
    } else if (ndac === 4) {
        dacCommand = sideLower === 'a' ? 0x62 : 0x72;
    } else {
        throw new Error('ndac analògic no vàlid.');
    }

    const data = [
        (value & 0xFF00) >> 8, 
        value & 0xFF
    ];

    await writeI2cBlock(i2cAddr, dacCommand, Buffer.from(data));

    const voltage = parseFloat(value) * 10 / 4095;
    console.log(`dacCommand: 0x${dacCommand.toString(16).toUpperCase()}`);
    console.log(`It has been sent 0x${value.toString(16).toUpperCase()} (${voltage.toFixed(2)} volts) to DAC ${ndac}`);
}

/**
 * Funció millorada amb readByte i neteja de buffer
 */
async function readByteDoubleImproved(i2cAddr, command) {
    if (!isBusOpen()) {
        throw new Error('El bus I2C no està obert.');
    }

    // Estratègia: Fes MOLTES lectures ràpides per netejar completament el buffer
    // Les primeres 3-4 lectures poden tenir dades antigues del buffer I2C
    
    let readings = [];
    const numReadings = 3; // Farem 5 lectures
    
    for (let i = 0; i < numReadings; i++) {
        const value = await new Promise((resolve, reject) => {
            i2cBus.readByte(i2cAddr, command, (err, res) => {
                if (err) return reject(err);
                resolve(res);
            });
        });
        
        readings.push(value);
        console.log(`Lectura ${i + 1}: 0x${value.toString(16).toUpperCase()}`);
        
        // Retard entre lectures
        await new Promise(resolve => setTimeout(resolve, 25));
    }
    
    // Agafa l'última lectura (la més fresca)
    const finalValue = readings[numReadings - 1];
    
    // Opcional: Validació - si les últimes 2 lectures coincideixen, és més fiable
    if (readings[numReadings - 1] === readings[numReadings - 2]) {
        console.log(`✓ Lectures consistents: 0x${finalValue.toString(16).toUpperCase()}`);
    } else {
        console.warn(`⚠ Lectures inconsistents: últimes 2 són 0x${readings[numReadings - 2].toString(16)} i 0x${finalValue.toString(16)}`);
    }

    return finalValue;
}

/**
 * Funció millorada amb FLUSH COMPLET del buffer I2C del kernel
 */
async function readByteWithKernelFlush(i2cAddr, command) {
    if (!isBusOpen()) {
        throw new Error('El bus I2C no està obert.');
    }

    // ESTRATÈGIA NOVA: Flush del buffer a nivell de kernel
    // Fem servir receiveBytes per buidar completament el buffer
    
    try {
        // 1. Intenta llegir bytes residuals del buffer (pot fallar, i està bé)
        const dummyBuffer = Buffer.alloc(32);
        try {
            await new Promise((resolve) => {
                i2cBus.i2cRead(i2cAddr, 32, dummyBuffer, () => {
                    // Ignora errors, només volem buidar el buffer
                    resolve();
                });
            });
        } catch (e) {
            // Ignora errors del flush
        }
        
        // 2. Retard per deixar que el buffer es netegi
        await new Promise(resolve => setTimeout(resolve, 50));
        
    } catch (e) {
        // Si falla el flush, continuem
    }

    // 3. Ara fem les lectures normals amb readByte
    let readings = [];
    const numReadings = 3;
    
    for (let i = 0; i < numReadings; i++) {
        const value = await new Promise((resolve, reject) => {
            i2cBus.readByte(i2cAddr, command, (err, res) => {
                if (err) return reject(err);
                resolve(res);
            });
        });
        
        readings.push(value);
        console.log(`Lectura ${i + 1}: 0x${value.toString(16).toUpperCase()}`);
        
        // Retard entre lectures
        await new Promise(resolve => setTimeout(resolve, 30));
    }
    
    // Validació: si les 3 lectures són diferents, hi ha problema
    const allSame = readings.every(val => val === readings[0]);
    const lastTwoSame = readings[1] === readings[2];
    
    if (allSame) {
        console.log(`✓ Les 3 lectures són idèntiques: 0x${readings[0].toString(16).toUpperCase()}`);
        return readings[0];
    } else if (lastTwoSame) {
        console.log(`✓ Les últimes 2 lectures coincideixen: 0x${readings[2].toString(16).toUpperCase()}`);
        return readings[2];
    } else {
        console.warn(`⚠ Lectures inconsistents: [${readings.map(r => '0x' + r.toString(16).toUpperCase()).join(', ')}]`);
        // Retorna l'última lectura però marca com a sospitosa
        return readings[2];
    }
}

/**
 * Funció simplificada que descarta les primeres lectures
 */
async function readByteCleanSimple(i2cAddr, command) {
    if (!isBusOpen()) {
        throw new Error('El bus I2C no està obert.');
    }

    // Fem 5 lectures en total, descartant les primeres 2
    let readings = [];
    // const totalReadings = 5;
    // const validReadings = 3; // Les últimes 3 són les vàlides
    const totalReadings = 2;
    const validReadings = 1; // Les últimes 3 són les vàlides
    
    for (let i = 0; i < totalReadings; i++) {
        const value = await new Promise((resolve, reject) => {
            i2cBus.readByte(i2cAddr, command, (err, res) => {
                if (err) return reject(err);
                resolve(res);
            });
        });
        
        readings.push(value);
        
        if (i < totalReadings - validReadings) {
            console.log(`Lectura ${i + 1} (descartada): 0x${value.toString(16).toUpperCase()}`);
        } else {
            console.log(`Lectura ${i + 1}: 0x${value.toString(16).toUpperCase()}`);
        }
        
        // Retard mínim entre lectures (optimitzat a 1ms)
        await new Promise(resolve => setTimeout(resolve, 1));
    }
    
    // Agafa les últimes 3 lectures
    const validValues = readings.slice(-validReadings);
    
    // Verifica si les últimes 2 o 3 coincideixen
    const allSame = validValues.every(val => val === validValues[0]);
    const lastTwoSame = validValues[1] === validValues[2];
    
    if (allSame) {
        console.log(`✓ Les 3 lectures vàlides són idèntiques: 0x${validValues[0].toString(16).toUpperCase()}`);
        return validValues[0];
    } else if (lastTwoSame) {
        console.log(`✓ Les últimes 2 lectures coincideixen: 0x${validValues[2].toString(16).toUpperCase()}`);
        return validValues[2];
    } else {
        console.warn(`⚠ Lectures inconsistents: [${validValues.map(r => '0x' + r.toString(16).toUpperCase()).join(', ')}] - Agafant l'última`);
        return validValues[2];
    }
}

/**
 * Llegeix una entrada digital (8 bits).
 * @param {string} addr Adreça binària (e.g., '0000').
 * @param {string} side Costat ('a' o 'b').
 * @returns {Promise<string>} Promesa que resol a la cadena binària de 8 bits.
 */

async function din(addr, side, doubleReading = true) {
    return queueOperation(async () => {
        const i2cAddr = getI2cAddr(addr, 'digital');
        if (!i2cAddr) {
            throw new Error('Adreça digital no vàlida.');
        }

        const sideLower = side.toLowerCase();
        const command = sideLower === 'a' ? 0x01 : 0x11;

        // Delay inicial eliminat per optimització (abans 2ms, anteriorment 10ms)
        
        let res;
        if (doubleReading) {
            // Mode doble lectura (per defecte): usa readByteCleanSimple que descarta la primera lectura
            res = await readByteCleanSimple(i2cAddr, command);
        } else {
            // Mode lectura simple: fa només 1 lectura directa
            res = await new Promise((resolve, reject) => {
                i2cBus.readByte(i2cAddr, command, (err, result) => {
                    if (err) return reject(err);
                    resolve(result);
                });
            });
        }
        
        const resp = res.toString(2).padStart(8, '0');
        return resp;
    });
}

/*
async function din(addr, side) {
    return queueOperation(async () => {
        const i2cAddr = getI2cAddr(addr, 'digital');
        if (!i2cAddr) {
            throw new Error('Adreça digital no vàlida.');
        }

        const sideLower = side.toLowerCase();
        const command = sideLower === 'a' ? 0x01 : 0x11;

        await new Promise(resolve => setTimeout(resolve, 10));
        
        // USA la nova funció simplificada
        const res = await readByteCleanSimple(i2cAddr, command);
        
        const resp = res.toString(2).padStart(8, '0');
        return resp;
    });
}
 

async function din(addr, side) {
    return queueOperation(async () => {
        const i2cAddr = getI2cAddr(addr, 'digital');
        if (!i2cAddr) {
            throw new Error('Adreça digital no vàlida.');
        }

        const sideLower = side.toLowerCase();
        const command = sideLower === 'a' ? 0x01 : 0x11;

        // Retard abans de començar la lectura
        // await new Promise(resolve => setTimeout(resolve, 30));
        await new Promise(resolve => setTimeout(resolve, 10));
        
        // USA readByteDoubleImproved en lloc de readByteDouble o readByteClean
        // const res = await readByteDoubleImproved(i2cAddr, command);
        const res = await readByteWithKernelFlush(i2cAddr, command);
        
        // Converteix a binari amb zfill(8)
        const resp = res.toString(2).padStart(8, '0');
        
        return resp;
    });
}

async function din(addr, side) {
    return queueOperation(async () => {
        const i2cAddr = getI2cAddr(addr, 'digital');
        if (!i2cAddr) {
            throw new Error('Adreça digital no vàlida.');
        }

        const sideLower = side.toLowerCase();
        const command = sideLower === 'a' ? 0x01 : 0x11;

        // Retard abans de la lectura
        await new Promise(resolve => setTimeout(resolve, 30));
        
        const res = await readByteDouble(i2cAddr, command);
        const resp = res.toString(2).padStart(8, '0');
        
        return resp;
    });
}


async function din(addr, side) {
    const i2cAddr = getI2cAddr(addr, 'digital');
    if (!i2cAddr) {
        throw new Error('Adreça digital no vàlida.');
    }

    const sideLower = side.toLowerCase();
    const command = sideLower === 'a' ? 0x01 : 0x11;

    const res = await readByteDouble(i2cAddr, command);

    // Conversió a binari amb zfill(8)
    const resp = res.toString(2).padStart(8, '0');
    return resp;
}
*/

/**
 * Escriu una sortida digital (8 bits).
 * @param {string} addr Adreça binària (e.g., '0000').
 * @param {string} side Costat ('a' o 'b').
 * @param {number} value Valor de 8 bits (0-255).
 * @returns {Promise<void>} Promesa que resol a l'èxit.
 */
async function dout(addr, side, value) {
    return queueOperation(async () => {
        const i2cAddr = getI2cAddr(addr, 'digital');
        if (!i2cAddr) {
            throw new Error('Adreça digital no vàlida.');
        }

        const sideLower = side.toLowerCase();
        const command = sideLower === 'a' ? 0x02 : 0x12;

        await writeI2cBlock(i2cAddr, command, Buffer.from([value]));
    });
}

/**
 * Escriu una sortida digital (8 bits) amb control PWM.
 * @param {string} addr Adreça binària (e.g., '0000').
 * @param {string} side Costat ('a' o 'b').
 * @param {number} value Valor de 8 bits (0-255).
 * @returns {Promise<void>} Promesa que resol a l'èxit.
 */
async function doutpwm(addr, side, value) {
    const i2cAddr = getI2cAddr(addr, 'digital');
    if (!i2cAddr) {
        throw new Error('Adreça digital no vàlida.');
    }

    const sideLower = side.toLowerCase();
    const command = sideLower === 'a' ? 0x03 : 0x13;

    await writeI2cBlock(i2cAddr, command, Buffer.from([value]));
}

/**
 * Escriu un sol bit de la sortida digital.
 * @param {string} addr Adreça binària.
 * @param {string} side Costat ('a' o 'b').
 * @param {number} posbyte Posició del bit (0-7).
 * @param {number} value Valor del bit (0 o 1).
 * @returns {Promise<void>} Promesa que resol a l'èxit.
 */
async function doutbit(addr, side, posbyte, value) {
    const i2cAddr = getI2cAddr(addr, 'digital');
    if (!i2cAddr) {
        throw new Error('Adreça digital no vàlida.');
    }

    let bitCfg;
    const sideLower = side.toLowerCase();
    const bitCfgsA = [0x0A, 0x2A, 0x4A, 0x6A, 0x8A, 0xAA, 0xCA, 0xEA];
    const bitCfgsB = [0x1A, 0x3A, 0x5A, 0x7A, 0x9A, 0xBA, 0xDA, 0xFA];

    if (sideLower === 'a') {
        bitCfg = bitCfgsA[posbyte];
    } else if (sideLower === 'b') {
        bitCfg = bitCfgsB[posbyte];
    }

    if (bitCfg === undefined) {
        throw new Error('Posició de bit o costat no vàlids.');
    }

    await writeI2cBlock(i2cAddr, bitCfg, Buffer.from([value]));
}

/**
 * Escriu un sol bit PWM de la sortida digital.
 * @param {string} addr Adreça binària.
 * @param {string} side Costat ('a' o 'b').
 * @param {number} posbyte Posició del bit (0-7).
 * @param {number} value Valor del PWM (0-255).
 * @returns {Promise<void>} Promesa que resol a l'èxit.
 */
async function doutbitpwm(addr, side, posbyte, value) {
    const i2cAddr = getI2cAddr(addr, 'digital');
    if (!i2cAddr) {
        throw new Error('Adreça digital no vàlida.');
    }

    let bitCfg;
    const sideLower = side.toLowerCase();
    const bitCfgsA = [0x0B, 0x2B, 0x4B, 0x6B, 0x8B, 0xAB, 0xCB, 0xEB];
    const bitCfgsB = [0x1B, 0x3B, 0x5B, 0x7B, 0x9B, 0xBB, 0xDB, 0xFB];

    if (sideLower === 'a') {
        bitCfg = bitCfgsA[posbyte];
    } else if (sideLower === 'b') {
        bitCfg = bitCfgsB[posbyte];
    }

    if (bitCfg === undefined) {
        throw new Error('Posició de bit o costat no vàlids.');
    }

    await writeI2cBlock(i2cAddr, bitCfg, Buffer.from([value]));
}

/**
 * Llegeix la versió del RIB Digital.
 * @param {string} addr Adreça binària.
 * @returns {Promise<object>} Promesa que resol a { major: number, minor: number, binary: string, log: string }.
 */
async function dversion(addr) {
    const i2cAddr = getI2cAddr(addr, 'digital');
    if (!i2cAddr) {
        throw new Error('Adreça digital no vàlida.');
    }

    const command = 0x06;
    const resp = await readWordDouble(i2cAddr, command);
    
    const major = (resp & 0xFF00) >> 8;
    const minor = resp & 0x00FF;

    const logStr = `Digital rib version: ${major}.${minor}`;
    console.log(logStr);

    const binResp = resp.toString(2).padStart(16, '0');
    return { major: major, minor: minor, binary: binResp, log: logStr };
}

/**
 * Llegeix la versió del RIB Analògic.
 * @param {string} addr Adreça binària.
 * @returns {Promise<object>} Promesa que resol a { major: number, minor: number, binary: string, log: string }.
 */
async function aversion(addr) {
    const i2cAddr = getI2cAddr(addr, 'analog');
    if (!i2cAddr) {
        throw new Error('Adreça analògica no vàlida.');
    }

    const command = 0x06;
    const resp = await readWordDouble(i2cAddr, command);
    
    const major = (resp & 0xFF00) >> 8;
    const minor = resp & 0x00FF;

    const logStr = `Analog rib version: ${major}.${minor}`;
    console.log(logStr);

    const binResp = resp.toString(2).padStart(16, '0');
    return { major: major, minor: minor, binary: binResp, log: logStr };
}

/**
 * Llegeix la configuració actual del RIB Digital.
 * @param {string} addr Adreça binària.
 * @returns {Promise<object>} Promesa que resol a { binary: string, modes: string[], log: string }.
 */
async function getdsetup(addr) {
    const i2cAddr = getI2cAddr(addr, 'digital');
    if (!i2cAddr) {
        throw new Error('Adreça digital no vàlida.');
    }

    const command = 0x05;
    const resp = await readByteDouble(i2cAddr, command);
    
    let output = [];
    if (resp & 0x01) output.push("A digital input");
    if (resp & 0x02) output.push("A digital output");
    if (resp & 0x04) output.push("A PWM output");
    if (resp & 0x10) output.push("B digital input");
    if (resp & 0x20) output.push("B digital output");
    if (resp & 0x40) output.push("B PWM output");
    if (resp & 0x80) output.push("B touch input");

    const printStr = output.join(', ');
    
    if (printStr.includes('B digital input')) {
        console.log(printStr.replace(', B digital input', ''));
        console.log('B digital input');
    } else if (printStr.includes('B digital output')) {
        console.log(printStr.replace(', B digital output', ''));
        console.log('B digital output');
    } else if (printStr.includes('B PWM output')) {
        console.log(printStr.replace(', B PWM output', ''));
        console.log('B PWM output');
    } else if (printStr.includes('B touch input')) {
        console.log(printStr.replace(', B touch input', ''));
        console.log('B touch input');
    } else {
        console.log(printStr);
    }
    
    const binResp = resp.toString(2).padStart(8, '0');
    return { binary: binResp, modes: output, log: printStr };
}

/**
 * Llegeix la configuració actual del RIB Analògic.
 * @param {string} addr Adreça binària.
 * @returns {Promise<object>} Promesa que resol a { binary: string, message: string, respA: number, respB: number }.
 */
async function getasetup(addr) {
    const i2cAddr = getI2cAddr(addr, 'analog');
    if (!i2cAddr) {
        throw new Error('Adreça analògica no vàlida.');
    }

    if (!isBusOpen()) {
        throw new Error('El bus I2C no està obert. Crida openBus() primer.');
    }

    // Funció auxiliar per lectura de byte sense tancar el bus
    const readByteSingle = (command) => new Promise(async (resolve, reject) => {
        try {
            // Primera lectura (es descarta)
            await new Promise((res, rej) => {
                i2cBus.readByte(i2cAddr, command, (err) => {
                    if (err) return rej(err);
                    res();
                });
            });
            
            // Segona lectura (vàlida)
            i2cBus.readByte(i2cAddr, command, (err, res2) => {
                if (err) return reject(err);
                resolve(res2);
            });
        } catch (err) {
            reject(err);
        }
    });

    const respA = await readByteSingle(0x25);
    const respB = await readByteSingle(0x35);

    console.log(`respA: 0x${respA.toString(16).toUpperCase()}, respB: 0x${respB.toString(16).toUpperCase()}`);
    
    let msgA, msgB;
    if (respA === 0x01) {
        msgA = "A rib is ADC, ";
    } else if (respA === 0x02) {
        msgA = "A rib is DAC, ";
    } else {
        msgA = "A rib is unconnected or it is unknown, ";
    }

    if (respB === 0x01) {
        msgB = "B rib is ADC.";
    } else if (respB === 0x02) {
        msgB = "B rib is DAC.";
    } else {
        msgB = "B rib is unconnected or it is unknown.";
    }

    const msg = msgA + msgB;
    console.log(msg);
    
    const resp = (respA * 16) | respB;
    const binResp = resp.toString(2).padStart(8, '0');
    
    // Construir array de modes
    const modes = [];
    if (respA === 0x01) {
        modes.push("A analog input (ADC)");
    } else if (respA === 0x02) {
        modes.push("A analog output (DAC)");
    } else {
        modes.push("A unconnected or unknown");
    }
    
    if (respB === 0x01) {
        modes.push("B analog input (ADC)");
    } else if (respB === 0x02) {
        modes.push("B analog output (DAC)");
    } else {
        modes.push("B unconnected or unknown");
    }
    
    return { 
        binary: binResp, 
        modes: modes, 
        description: msg,
        respA: respA, 
        respB: respB 
    };
}

/**
 * Estableix la configuració del RIB Digital.
 * @param {string} addr Adreça binària.
 * @param {string} modeA Mode per al costat A (ain, aout, aoutpwm).
 * @param {string} modeB Mode per al costat B (bin, bout, boutpwm, bintouch).
 * @returns {Promise<boolean>} Promesa que resol a true si es va canviar la configuració.
 */
async function dsetup(addr, modeA, modeB) {
    const i2cAddr = getI2cAddr(addr, 'digital');
    if (!i2cAddr) {
        throw new Error('Adreça digital no vàlida.');
    }

    if (!isBusOpen()) {
        throw new Error('El bus I2C no està obert. Crida openBus() primer.');
    }

    let modeL = 0x00;
    let modeH = 0x00;
    const modeALower = modeA.toLowerCase();
    const modeBLower = modeB.toLowerCase();

    if (modeALower === 'ain') { 
        modeH |= 0xFF;
    } else if (modeALower === 'aout' || modeALower === 'aoutpwm') {
        modeH |= 0x00;
    }

    if (modeBLower === 'bin' || modeBLower === 'bintouch') {
        modeL |= 0xFF;
    } else if (modeBLower === 'bout' || modeBLower === 'boutpwm') {
        modeL |= 0x00;
    }
    
    console.log(`${modeALower}, ${modeBLower}`);

    // Funció auxiliar per escriure blocs sense tancar el bus
    const writeBlock = (command, data) => new Promise((resolve, reject) => {
        i2cBus.writeI2cBlock(i2cAddr, command, data.length, Buffer.from(data), (err) => {
            if (err) return reject(err);
            resolve();
        });
    });
    
    let operations = [];
    let success = false;

    if ((modeALower === 'ain' || modeALower === 'aout') && (modeBLower === 'bin' || modeBLower === 'bout')) {
        operations.push(() => writeBlock(0x00, [modeL, modeH]));
        success = true;
    } else if ((modeALower === 'ain' || modeALower === 'aout') && (modeBLower === 'boutpwm')) {
        operations.push(() => writeBlock(0x00, [0x00, modeH]));
        operations.push(() => writeBlock(0x04, [0x01]));
        success = true;
    } else if ((modeALower === 'aoutpwm') && (modeBLower === 'bin' || modeBLower === 'bout')) {
        operations.push(() => writeBlock(0x00, [modeL, 0x00]));
        operations.push(() => writeBlock(0x04, [0x00]));
        success = true;
    } else if ((modeALower === 'ain' || modeALower === 'aout' || modeALower === 'aoutpwm') && (modeBLower === 'bintouch')) {
        operations.push(() => writeBlock(0x00, [0xFF, modeH]));
        if (modeALower === 'aoutpwm') {
            operations.push(() => writeBlock(0x04, [0x00]));
        }
        operations.push(() => writeBlock(0x07, [0x01]));
        success = true;
    } else {
        console.log('Setup has not been changed');
        success = false;
    }

    // Executa les operacions I2C en seqüència
    for (const op of operations) {
        await op();
    }

    return success;
}

// --- Exporta les funcions ---
/**
 * Comprova si el bus I2C està actualment en ús per una operació.
 * @returns {boolean} True si hi ha una operació en curs, false si està lliure
 */
function isBusOperating() {
    return isOperating;
}

/**
 * Espera activament fins que el bus I2C estigui lliure.
 * Útil per evitar col·lisions quan múltiples nodes intenten accedir simultàniament.
 * 
 * @param {number} timeoutMs - Temps màxim d'espera en mil·lisegons (per defecte: 1000ms)
 * @returns {Promise<number>} - Retorna el temps d'espera en ms (0 si estava lliure)
 * @throws {Error} - Si s'exhaureix el timeout abans que el bus estigui lliure
 */
function waitForBusFree(timeoutMs = 1000) {
    return new Promise((resolve, reject) => {
        const startTime = Date.now();
        const pollInterval = 5; // Comprova cada 5ms
        
        function checkBus() {
            const elapsed = Date.now() - startTime;
            
            if (!isOperating) {
                // Bus lliure!
                resolve(elapsed);
            } else if (elapsed >= timeoutMs) {
                // Timeout exhaurit
                reject(new Error(`Timeout esperant que el bus I2C estigui lliure (${timeoutMs}ms)`));
            } else {
                // Continua esperant
                setTimeout(checkBus, pollInterval);
            }
        }
        
        // Inicia la comprovació
        checkBus();
    });
}

module.exports = {
    openBus,
    closeBus,
    isBusOpen,
    isBusOperating,
    waitForBusFree,
    ain,
    ain2v,
    ainv,
    aout,
    v2aout,
    din,
    dout,
    doutpwm,
    doutbit,
    doutbitpwm,
    dversion,
    aversion,
    getdsetup,
    getasetup,
    dsetup
};

// --- Exemple d'ús ---
/*
if (require.main === module) {
    const iotv4 = require('./i2c-iotv4');
    
    async function runExample() {
        try {
            console.log("=== Exemple d'ús de i2c-iotv4.js ===\n");

            // 1. Obre el bus I2C al principi
            console.log("1. Obrint el bus I2C...");
            await iotv4.openBus();
            console.log("   Bus I2C obert correctament.\n");

            // 2. Llegeix la versió digital
            console.log("2. Llegint versió del RIB Digital...");
            const version = await iotv4.dversion('0000');
            console.log(`   Versió: ${version.major}.${version.minor}\n`);

            // 3. Escriu al DAC 1 (aprox. 5V)
            console.log("3. Escrivint 5V al DAC 1 del costat A...");
            const dacValue = 2048;
            await iotv4.aout('0000', 'a', 1, dacValue);
            console.log("   Escriptura completada.\n");
            
            // 4. Llegeix l'ADC 1
            console.log("4. Llegint ADC 1 del costat A...");
            const adcReading = await iotv4.ainv('0000', 'a', 1);
            console.log(`   Valor raw: ${adcReading.raw}`);
            console.log(`   Voltatge: ${adcReading.voltage.toFixed(2)} V\n`);

            // 5. Prova entrada/sortida digital
            console.log("5. Llegint entrada digital del costat A...");
            const digitalInput = await iotv4.din('0000', 'a');
            console.log(`   Valor binari: ${digitalInput}\n`);

            // 6. Escriu sortida digital
            console.log("6. Escrivint 0xFF a la sortida digital del costat B...");
            await iotv4.dout('0000', 'b', 0xFF);
            console.log("   Escriptura completada.\n");

            // 7. Tanca el bus I2C al final
            console.log("7. Tancant el bus I2C...");
            await iotv4.closeBus();
            console.log("   Bus I2C tancat correctament.\n");

            console.log("=== Exemple completat amb èxit ===");

        } catch (error) {
            console.error("\n!!! ERROR !!!");
            console.error(`Missatge: ${error.message}`);
            console.error("\nAssegura't que:");
            console.error("- El RIB està connectat correctament");
            console.error("- L'I2C està activat a la Raspberry Pi (raspi-config)");
            console.error("- Tens els permisos necessaris per accedir a /dev/i2c-1");
            console.error("- El paquet 'i2c-bus' està instal·lat (npm install i2c-bus)");
            
            // Intenta tancar el bus si està obert
            if (iotv4.isBusOpen()) {
                try {
                    await iotv4.closeBus();
                    console.error("\nEl bus I2C s'ha tancat després de l'error.");
                } catch (closeError) {
                    console.error("\nNo s'ha pogut tancar el bus I2C.");
                }
            }
        }
    }
    
    runExample();
}
*/
