/**
 * Node-RED node: getmplc
 * Llegeix bits d'una variable de memòria M i retorna el valor decimal
 */

module.exports = function(RED) {
    function GetMPlcNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        // Configuració del node
        const defaultAddr = config.addr || "0000";
        const defaultSide = config.side || "A";

        node.on('input', function(msg) {
            try {
                // Determinar addr i side
                let addr = defaultAddr;
                let side = defaultSide;
                
                // Prioritat: msg.topic amb JSON té prioritat sobre configuració
                if (msg.topic && typeof msg.topic === 'string') {
                    try {
                        const topicData = JSON.parse(msg.topic);
                        if (topicData.addr) addr = topicData.addr;
                        if (topicData.side) side = topicData.side;
                    } catch (e) {
                        // Si no es pot parsejar, ignora i usa configuració
                    }
                }
                
                // Si msg.payload és un booleà true, activa la lectura
                if (msg.payload === true || msg.payload === undefined) {
                    // Converteix l'adreça binària a decimal
                    const addrDecimal = parseInt(addr, 2);
                    if (isNaN(addrDecimal) || addrDecimal < 0 || addrDecimal > 15) {
                        node.error('Adreça invàlida: ' + addr);
                        node.status({fill: "red", shape: "dot", text: "error: adreça invàlida"});
                        return;
                    }

                    // Determina el nom de la variable M
                    const varname = "M" + addrDecimal;
                    
                    // Obté la variable del context global
                    const store = node.context().global;
                    const ctxvar = store.get(varname);

                    // Si la variable no existeix, retorna error
                    if (ctxvar === undefined) {
                        node.error(`Variable ${varname} no existeix`);
                        node.status({fill: "red", shape: "dot", text: `error: ${varname} no existeix`});
                        msg.payload = {
                            addr: addr,
                            side: side,
                            error: `Variable ${varname} no existeix`
                        };
                        node.send(msg);
                        return;
                    }

                    // Determina l'offset segons el costat (A=0, B=10)
                    const sideLower = side.toLowerCase();
                    let offset = 0;
                    
                    if (sideLower === 'a') {
                        offset = 0;
                    } else if (sideLower === 'b') {
                        offset = 10;
                    } else {
                        node.error('Side ha de ser A o B');
                        node.status({fill: "red", shape: "dot", text: "error: side invàlid"});
                        return;
                    }

                    // Llegeix cada bit de la variable M i construeix el valor
                    let value = 0;
                    for (let bitPos = 0; bitPos < 8; bitPos++) {
                        const memIndex = offset + bitPos;
                        
                        // Comprova que no sortim dels límits de l'array
                        if (memIndex < 64 && ctxvar[memIndex] === true) {
                            value |= (1 << bitPos); // Activa el bit corresponent
                        }
                    }

                    // Prepara la sortida
                    msg.payload = {
                        addr: addr,
                        side: side,
                        value: value
                    };

                    // Mostra l'estat
                    const hexValue = value.toString(16).toUpperCase().padStart(2, '0');
                    const statusText = `${varname}.${offset}-${offset+7}: 0x${hexValue} (${value})`;
                    node.status({fill: "green", shape: "dot", text: statusText});

                    node.send(msg);
                    
                } else {
                    node.warn('El payload ha de ser true per activar la lectura, o usa msg.topic amb JSON de configuració');
                    node.status({fill: "yellow", shape: "dot", text: "esperant activació"});
                }
                
            } catch (error) {
                node.error(`Error en getmplc: ${error.message}`);
                node.status({fill: "red", shape: "ring", text: "error"});
                msg.payload = { error: error.message };
                node.send(msg);
            }
        });
    }

    RED.nodes.registerType("getmplc", GetMPlcNode);
}

