/**
 * Node-RED node: getmac
 * Llegeix les adreces MAC de les interfícies Ethernet i WiFi
 */

module.exports = function(RED) {
    const fs = require('fs');
    const { execSync } = require('child_process');

    function GetMacNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;

        // Funció per llegir la MAC d'una interfície
        function getMacAddress(interface) {
            try {
                // Mètode 1: Llegir del fitxer /sys/class/net
                const macPath = `/sys/class/net/${interface}/address`;
                if (fs.existsSync(macPath)) {
                    const mac = fs.readFileSync(macPath, 'utf8').trim();
                    // Convertir a majúscules i eliminar els dos punts
                    return mac.toUpperCase().replace(/:/g, '');
                }
                
                // Mètode 2: Usar ip link show com a fallback
                try {
                    const output = execSync(`ip link show ${interface}`, { encoding: 'utf8' });
                    const macMatch = output.match(/link\/ether\s+([0-9a-f:]+)/i);
                    if (macMatch && macMatch[1]) {
                        return macMatch[1].toUpperCase().replace(/:/g, '');
                    }
                } catch (e) {
                    // Si falla, continuar amb l'error principal
                }
                
                throw new Error(`Interfície ${interface} no trobada`);
                
            } catch (error) {
                throw error;
            }
        }

        node.on('input', function(msg) {
            try {
                node.status({fill: "blue", shape: "dot", text: "llegint MAC..."});

                // Llegir MAC Ethernet (sortida 1)
                let msgEthernet = { ...msg };
                try {
                    const ethMac = getMacAddress('eth0');
                    msgEthernet.payload = ethMac;
                    msgEthernet.interface = 'eth0';
                    msgEthernet.type = 'ethernet';
                } catch (error) {
                    msgEthernet.payload = null;
                    msgEthernet.interface = 'eth0';
                    msgEthernet.type = 'ethernet';
                    msgEthernet.error = error.message;
                    node.warn(`Error llegint MAC Ethernet: ${error.message}`);
                }

                // Llegir MAC WiFi (sortida 2)
                let msgWifi = { ...msg };
                try {
                    const wifiMac = getMacAddress('wlan0');
                    msgWifi.payload = wifiMac;
                    msgWifi.interface = 'wlan0';
                    msgWifi.type = 'wifi';
                } catch (error) {
                    msgWifi.payload = null;
                    msgWifi.interface = 'wlan0';
                    msgWifi.type = 'wifi';
                    msgWifi.error = error.message;
                    node.warn(`Error llegint MAC WiFi: ${error.message}`);
                }

                // Mostrar estat
                const ethDisplay = msgEthernet.payload ? msgEthernet.payload.substring(0, 8) + '...' : 'N/A';
                const wifiDisplay = msgWifi.payload ? msgWifi.payload.substring(0, 8) + '...' : 'N/A';
                node.status({fill: "green", shape: "dot", text: `ETH: ${ethDisplay} | WiFi: ${wifiDisplay}`});

                // Enviar missatges a les dues sortides
                node.send([msgEthernet, msgWifi]);

            } catch (error) {
                node.error(`Error en getmac: ${error.message}`);
                node.status({fill: "red", shape: "ring", text: "error"});
                
                // Enviar errors a les dues sortides
                let errorMsg1 = { ...msg, payload: null, error: error.message };
                let errorMsg2 = { ...msg, payload: null, error: error.message };
                node.send([errorMsg1, errorMsg2]);
            }
        });
    }

    RED.nodes.registerType("getmac", GetMacNode);
}
