/**
 * Node-RED node: dsetup
 * Estableix la configuració del RIB Digital
 */

module.exports = function(RED) {
    const { dsetup } = require('./i2c-iotv4');

    function DsetupNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        // Configuració del node
        const defaultAddr = config.addr || "0000";
        const defaultModeA = config.modeA || "ain";
        const defaultModeB = config.modeB || "bout";

        node.on('input', async function(msg) {
            try {
                let addr = defaultAddr;
                let modeA = defaultModeA;
                let modeB = defaultModeB;
                
                // Prioritat: si payload és un objecte amb els camps, usar-los
                if (msg.payload && typeof msg.payload === 'object' && !Array.isArray(msg.payload)) {
                    if (msg.payload.addr) addr = msg.payload.addr;
                    if (msg.payload.modeA) modeA = msg.payload.modeA;
                    if (msg.payload.modeB) modeB = msg.payload.modeB;
                }
                // Si payload és true, false o timestamp, usar configuració del node
                // (que ja està assignada per defecte)

                // Validació dels modes
                const validModesA = ['ain', 'aout', 'aoutpwm'];
                const validModesB = ['bin', 'bout', 'boutpwm', 'bintouch'];

                if (!validModesA.includes(modeA.toLowerCase())) {
                    node.error(`modeA invàlid. Opcions vàlides: ${validModesA.join(', ')}`);
                    node.status({fill: "red", shape: "dot", text: "error: modeA invàlid"});
                    return;
                }

                if (!validModesB.includes(modeB.toLowerCase())) {
                    node.error(`modeB invàlid. Opcions vàlides: ${validModesB.join(', ')}`);
                    node.status({fill: "red", shape: "dot", text: "error: modeB invàlid"});
                    return;
                }

                node.status({fill: "blue", shape: "dot", text: "configurant..."});

                // Crida a la funció dsetup
                const success = await dsetup(addr, modeA, modeB);

                // Prepara la sortida
                msg.payload = {
                    addr: addr,
                    modeA: modeA,
                    modeB: modeB,
                    success: success,
                    message: success ? 'Configuració establerta correctament' : 'La configuració no s\'ha canviat'
                };

                if (success) {
                    node.status({fill: "green", shape: "dot", text: `${modeA}, ${modeB}`});
                } else {
                    node.status({fill: "yellow", shape: "dot", text: "no canviat"});
                }
                
                node.send(msg);

            } catch (error) {
                node.error(`Error en dsetup: ${error.message}`);
                node.status({fill: "red", shape: "ring", text: "error"});
                msg.payload = { error: error.message };
                node.send(msg);
            }
        });
    }

    RED.nodes.registerType("dsetup", DsetupNode);
}
