/**
 * Node-RED node: doutbit
 * Escriu un sol bit de la sortida digital
 */

module.exports = function(RED) {
    const { doutbit } = require('./i2c-iotv4');

    function DoutbitNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        // Configuració del node
        const defaultAddr = config.addr || "0000";
        const defaultSide = config.side || "A";
        const defaultBit = parseInt(config.bit) || 0;

        node.on('input', async function(msg) {
            try {
                // Determinar addr, side, posbyte i value
                let addr = defaultAddr;
                let side = defaultSide;
                let posbyte = defaultBit;
                let value = null;
                let useLegacyMode = false;
                
                // Prioritat 1: msg.topic amb JSON té prioritat sobre tot
                if (msg.topic && typeof msg.topic === 'string') {
                    try {
                        const topicData = JSON.parse(msg.topic);
                        if (topicData.addr && topicData.side && 
                            topicData.posbyte !== undefined && topicData.value !== undefined) {
                            addr = topicData.addr;
                            side = topicData.side;
                            posbyte = topicData.posbyte;
                            value = topicData.value;
                        }
                    } catch (e) {
                        // Si no es pot parsejar, continua amb altres opcions
                    }
                }
                
                // Prioritat 2: Si payload és un booleà
                if (value === null && typeof msg.payload === 'boolean') {
                    value = msg.payload ? 1 : 0;
                }
                
                // Prioritat 3: Si payload és un objecte amb tots els camps (mode llegat)
                if (value === null && msg.payload && typeof msg.payload === 'object' && !Array.isArray(msg.payload)) {
                    if (msg.payload.addr && msg.payload.side && 
                        msg.payload.posbyte !== undefined && msg.payload.value !== undefined) {
                        addr = msg.payload.addr;
                        side = msg.payload.side;
                        posbyte = msg.payload.posbyte;
                        value = msg.payload.value;
                        useLegacyMode = true;
                    }
                }
                
                // Si tenim un valor per escriure, procedir
                if (value !== null) {
                    // Validacions
                    if (typeof posbyte !== 'number' || posbyte < 0 || posbyte > 7) {
                        node.error('posbyte ha de ser un nombre entre 0 i 7');
                        node.status({fill: "red", shape: "dot", text: "error: posbyte invàlid"});
                        return;
                    }

                    if (typeof value !== 'number' || (value !== 0 && value !== 1)) {
                        node.error('value ha de ser 0 o 1');
                        node.status({fill: "red", shape: "dot", text: "error: value invàlid"});
                        return;
                    }

                    node.status({fill: "blue", shape: "dot", text: `escrivint bit ${posbyte}=${value}...`});

                    // Crida a la funció doutbit
                    await doutbit(addr, side, posbyte, value);

                    // Prepara la sortida
                    msg.payload = {
                        addr: addr,
                        side: side,
                        posbyte: posbyte,
                        value: value,
                        success: true
                    };

                    node.status({fill: "green", shape: "dot", text: `bit ${posbyte}=${value}`});
                    node.send(msg);
                    
                } else {
                    node.warn('El payload ha de ser true/false per escriure un bit, o usa msg.topic amb JSON de configuració');
                    node.status({fill: "yellow", shape: "dot", text: "esperant activació"});
                }

            } catch (error) {
                node.error(`Error en doutbit: ${error.message}`);
                node.status({fill: "red", shape: "ring", text: "error"});
                msg.payload = { error: error.message };
                node.send(msg);
            }
        });
    }

    RED.nodes.registerType("doutbit", DoutbitNode);
}

