/**
 * Node-RED node: dout
 * Escriu una sortida digital (8 bits)
 */

module.exports = function(RED) {
    const { dout } = require('./i2c-iotv4');

    function DoutNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        // Configuració del node
        const defaultAddr = config.addr || "0000";
        const defaultSide = config.side || "A";

        node.on('input', async function(msg) {
            try {
                // Determinar addr, side i value
                let addr = defaultAddr;
                let side = defaultSide;
                let value = null;
                let useLegacyMode = false;
                
                // Prioritat 1: msg.topic amb JSON té prioritat sobre tot
                if (msg.topic && typeof msg.topic === 'string') {
                    try {
                        const topicData = JSON.parse(msg.topic);
                        if (topicData.addr && topicData.side && topicData.value !== undefined) {
                            addr = topicData.addr;
                            side = topicData.side;
                            value = topicData.value;
                        }
                    } catch (e) {
                        // Si no es pot parsejar, continua amb altres opcions
                    }
                }
                
                // Prioritat 2: Si payload és un número
                if (value === null && typeof msg.payload === 'number') {
                    value = msg.payload;
                }
                
                // Prioritat 3: Si payload és un objecte amb addr, side i value (mode llegat)
                if (value === null && msg.payload && typeof msg.payload === 'object' && !Array.isArray(msg.payload)) {
                    if (msg.payload.addr && msg.payload.side && msg.payload.value !== undefined) {
                        addr = msg.payload.addr;
                        side = msg.payload.side;
                        value = msg.payload.value;
                        useLegacyMode = true;
                    }
                }
                
                // Si tenim un valor per escriure, procedir
                if (value !== null) {
                    // Validació del valor
                    if (typeof value !== 'number' || value < 0 || value > 255) {
                        node.error('El valor ha de ser un nombre entre 0 i 255');
                        node.status({fill: "red", shape: "dot", text: "error: valor invàlid"});
                        return;
                    }

                    // Converteix a enter per si ve com a float
                    value = Math.floor(value);

                    node.status({fill: "blue", shape: "dot", text: `escrivint 0x${value.toString(16).toUpperCase()}...`});

                    // Crida a la funció dout
                    await dout(addr, side, value);

                    // Prepara la sortida
                    msg.payload = {
                        addr: addr,
                        side: side,
                        value: value,
                        hex: `0x${value.toString(16).toUpperCase()}`,
                        binary: value.toString(2).padStart(8, '0'),
                        success: true
                    };

                    node.status({fill: "green", shape: "dot", text: `0x${value.toString(16).toUpperCase()}`});
                    node.send(msg);
                    
                } else {
                    node.warn('El payload ha de ser un número (0-255) per escriure, o usa msg.topic amb JSON de configuració');
                    node.status({fill: "yellow", shape: "dot", text: "esperant activació"});
                }

            } catch (error) {
                node.error(`Error en dout: ${error.message}`);
                node.status({fill: "red", shape: "ring", text: "error"});
                msg.payload = { error: error.message };
                node.send(msg);
            }
        });
    }

    RED.nodes.registerType("dout", DoutNode);
}

