/**
 * Node-RED node: dinbit
 * Llegeix una entrada digital (8 bits) i separa cada bit en sortides individuals
 */

module.exports = function(RED) {
    const { din } = require('./i2c-iotv4');

    function DinBitNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        // Configuració del node
        const defaultAddr = config.addr || "0000";
        const defaultSide = config.side || "A";

        node.on('input', async function(msg) {
            const maxRetries = 2;
            let attempt = 0;
            
            // Determinar addr i side
            let addr = defaultAddr;
            let side = defaultSide;
            
            // Prioritat: msg.topic amb JSON té prioritat sobre configuració
            if (msg.topic && typeof msg.topic === 'string') {
                try {
                    const topicData = JSON.parse(msg.topic);
                    if (topicData.addr) addr = topicData.addr;
                    if (topicData.side) side = topicData.side;
                } catch (e) {
                    // Si no es pot parsejar, ignora i usa configuració
                }
            }
            
            // Si msg.payload és un booleà true, activa la lectura
            if (msg.payload === true || msg.payload === undefined) {
                // Realitza la lectura
                while (attempt < maxRetries) {
                    try {
                        node.status({fill: "blue", shape: "dot", text: `llegint (intent ${attempt + 1})...`});

                        await new Promise(resolve => setTimeout(resolve, 5));
                        
                        const result = await din(addr, side);
                        const decimalValue = parseInt(result, 2);

                        // Extreu cada bit i crea missatges per a cada sortida
                        const messages = [];
                        for (let bitPos = 0; bitPos < 8; bitPos++) {
                            const stringIndex = result.length - 1 - bitPos;
                            const bitChar = result[stringIndex];
                            const bitValue = bitChar === '1' ? true : false;
                            
                            messages.push({
                                payload: bitValue,
                                topic: `bit${bitPos}`,
                                addr: addr,
                                side: side,
                                binary: result,
                                decimal: decimalValue,
                                hex: `0x${decimalValue.toString(16).toUpperCase()}`
                            });
                        }

                        node.status({fill: "green", shape: "dot", text: result});
                        node.send(messages);
                        return; // Sortida exitosa

                    } catch (error) {
                        attempt++;
                        if (attempt >= maxRetries) {
                            node.error(`Error en dinbit després de ${maxRetries} intents: ${error.message}`);
                            node.status({fill: "red", shape: "ring", text: "error"});
                            
                            // Envia error a totes les sortides
                            const errorMessages = [];
                            for (let i = 0; i < 8; i++) {
                                errorMessages.push({
                                    payload: null,
                                    error: error.message,
                                    attempts: attempt
                                });
                            }
                            node.send(errorMessages);
                        } else {
                            await new Promise(resolve => setTimeout(resolve, 100));
                        }
                    }
                }
            } else {
                node.warn('El payload ha de ser true per activar la lectura, o usa msg.topic amb JSON de configuració');
                node.status({fill: "yellow", shape: "dot", text: "esperant activació"});
            }
        });
    }

    RED.nodes.registerType("dinbit", DinBitNode);
}

