/**
 * Node-RED node: din
 * Llegeix una entrada digital (8 bits)
 */

module.exports = function(RED) {
    const { din } = require('./i2c-iotv4');

    function DinNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        // Configuració del node
        const defaultAddr = config.addr || "0000";
        const defaultSide = config.side || "A";
        const doubleReading = config.doubleReading !== undefined ? config.doubleReading : true;

        node.on('input', async function(msg) {
            const maxRetries = 2;
            let attempt = 0;
            
            // Determinar addr i side
            let addr = defaultAddr;
            let side = defaultSide;
            let useLegacyMode = false;
            
            // Prioritat 1: msg.topic amb JSON té prioritat sobre tot
            if (msg.topic && typeof msg.topic === 'string') {
                try {
                    const topicData = JSON.parse(msg.topic);
                    if (topicData.addr && topicData.side) {
                        addr = topicData.addr;
                        side = topicData.side;
                    }
                } catch (e) {
                    // Si no es pot parsejar, continua amb altres opcions
                }
            }
            
            // Prioritat 2: Si payload és un objecte amb addr i side (mode llegat)
            if (msg.payload && typeof msg.payload === 'object' && !Array.isArray(msg.payload)) {
                if (msg.payload.addr && msg.payload.side) {
                    addr = msg.payload.addr;
                    side = msg.payload.side;
                    useLegacyMode = true;
                }
            }
            
            // Si msg.payload és un booleà true o mode llegat, activa la lectura
            if (msg.payload === true || useLegacyMode || msg.payload === undefined) {
                // Realitza la lectura
                while (attempt < maxRetries) {
                    try {
                        // node.status({fill: "blue", shape: "dot", text: `llegint (intent ${attempt + 1})...`});

                        // Petit retard abans de la lectura
                        // await new Promise(resolve => setTimeout(resolve, 5));
                        await new Promise(resolve => setTimeout(resolve, 3));
                        
                        const result = await din(addr, side, doubleReading);
                        const decimalValue = parseInt(result, 2);

                        msg.payload = {
                            addr: addr,
                            side: side,
                            binary: result,
                            decimal: decimalValue,
                            hex: `0x${decimalValue.toString(16).toUpperCase()}`,
                            attempts: attempt + 1,
                            doubleReading: doubleReading
                        };

                        node.status({fill: "green", shape: "dot", text: result});
                        node.send(msg);
                        return; // Sortida exitosa

                    } catch (error) {
                        attempt++;
                        if (attempt >= maxRetries) {
                            node.error(`Error en din després de ${maxRetries} intents: ${error.message}`);
                            node.status({fill: "red", shape: "ring", text: "error"});
                            msg.payload = { error: error.message, attempts: attempt };
                            node.send(msg);
                        } else {
                            // Espera abans de reintentar
                            await new Promise(resolve => setTimeout(resolve, 100));
                        }
                    }
                }
            } else {
                node.warn('El payload ha de ser true per activar la lectura, o usa msg.topic amb JSON de configuració');
                node.status({fill: "yellow", shape: "dot", text: "esperant activació"});
            }
        });
    }

    RED.nodes.registerType("din", DinNode);
}

