/**
 * Node-RED node: cpdtitoi
 * Copia els valors de les variables DTI (Digital Twin Input) a les variables I (Input)
 * Permet sincronitzar entrades virtuals del bessó digital amb variables d'entrada físiques
 */

module.exports = function(RED) {
    
    // Funció per inicialitzar variables I si no existeixen
    function initVariableI(node, varname) {
        const store = node.context().global;
        const N_INDEX = 64;
        const ctxvar = Array(N_INDEX).fill(false);
        store.set(varname, ctxvar);
        return ctxvar;
    }

    function CpDtiToINode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        node.iserror = false;

        node.on('input', function(msg) {
            try {
                node.status({fill: "blue", shape: "dot", text: "copiant variables..."});

                const store = node.context().global;
                const copiedVariables = [];
                let copyCount = 0;

                // Busca totes les variables que comencen amb "DTI"
                // Node-RED context global no té un mètode per llistar claus,
                // així que intentarem les variables DTI de 0 a 999
                for (let address = 0; address <= 999; address++) {
                    const dtiVarname = "DTI" + address;
                    const iVarname = "I" + address;
                    
                    // Comprova si la variable DTI existeix
                    const dtiVar = store.get(dtiVarname);
                    
                    if (dtiVar !== undefined && Array.isArray(dtiVar)) {
                        // Obté o crea la variable I corresponent
                        let iVar = store.get(iVarname);
                        
                        if (iVar === undefined) {
                            node.warn(`Variable ${iVarname} no existeix, creant-la...`);
                            iVar = initVariableI(node, iVarname);
                        }
                        
                        // Copia tots els valors de DTI a I
                        for (let index = 0; index < 64; index++) {
                            iVar[index] = dtiVar[index] ? true : false;
                        }
                        
                        // Actualitza la variable I al context global
                        store.set(iVarname, iVar);
                        
                        // Registra la còpia
                        copiedVariables.push(`${dtiVarname} → ${iVarname}`);
                        copyCount++;
                    }
                }

                // Prepara la sortida
                msg.payload = {
                    copied: copiedVariables,
                    count: copyCount,
                    timestamp: Date.now(),
                    success: true
                };

                if (copyCount === 0) {
                    msg.payload.message = "No s'han trobat variables DTI per copiar";
                    node.status({fill: "yellow", shape: "dot", text: "cap variable DTI trobada"});
                } else {
                    // Mostra l'estat
                    const statusText = `${copyCount} variable${copyCount !== 1 ? 's' : ''} copiada${copyCount !== 1 ? 'es' : ''}`;
                    node.status({fill: "green", shape: "dot", text: statusText});
                }

                // Reenvia el missatge
                node.send(msg);

            } catch (error) {
                node.error(`Error en cpdtitoi: ${error.message}`);
                node.status({fill: "red", shape: "ring", text: "error"});
                msg.payload = { 
                    error: error.message,
                    success: false,
                    timestamp: Date.now()
                };
                node.send(msg);
            }
        });
    }

    RED.nodes.registerType("cpdtitoi", CpDtiToINode);
}
