/**
 * closebus.js
 * Node de Node-RED per tancar el bus I2C
 */

module.exports = function(RED) {
    const iotv4 = require('./i2c-iotv4');

    function CloseBusNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;

        node.on('input', async function(msg) {
            try {
                // Comprova si el bus està obert
                if (!iotv4.isBusOpen()) {
                    node.warn('El bus I2C no està obert');
                    msg.payload = {
                        success: false,
                        message: 'El bus I2C no està obert',
                        alreadyClosed: true
                    };
                    node.send(msg);
                    return;
                }

                // Tanca el bus I2C
                await iotv4.closeBus();
                
                node.status({fill: "grey", shape: "ring", text: "Bus tancat"});
                
                msg.payload = {
                    success: true,
                    message: 'Bus I2C tancat correctament',
                    timestamp: new Date().toISOString()
                };
                
                node.send(msg);
                
            } catch (error) {
                node.status({fill: "red", shape: "ring", text: "Error"});
                node.error(`Error tancant el bus I2C: ${error.message}`, msg);
                
                msg.payload = {
                    success: false,
                    error: error.message
                };
                
                node.send(msg);
            }
        });

        node.on('close', function(done) {
            node.status({});
            // Intenta tancar el bus si està obert quan es tanca el node
            if (iotv4.isBusOpen()) {
                iotv4.closeBus()
                    .then(() => done())
                    .catch(() => done());
            } else {
                done();
            }
        });
    }

    RED.nodes.registerType("closebus", CloseBusNode);
};
