/**
 * Node-RED node: aout
 * Escriu una sortida analògica (0-10V) al DAC
 */

module.exports = function(RED) {
    const { aout, v2aout } = require('./i2c-iotv4');

    function AoutNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        // Configuració del node
        const defaultAddr = config.addr || "0000";
        const defaultSide = config.side || "A";
        const defaultChannel = parseInt(config.channel) || 1;
        const defaultVoltage = parseFloat(config.voltage) || 5.0;
        const defaultSaveEeprom = config.saveEeprom || false;

        node.on('input', async function(msg) {
            try {
                // Determinar addr, side, channel, voltage i saveEeprom
                let addr = defaultAddr;
                let side = defaultSide;
                let channel = defaultChannel;
                let voltage = null;
                let saveEeprom = defaultSaveEeprom;
                let voltageSource = "default"; // Per debugging
                
                // Prioritat 1: msg.topic amb JSON té prioritat sobre tot
                if (msg.topic && typeof msg.topic === 'string') {
                    try {
                        const topicData = JSON.parse(msg.topic);
                        if (topicData.addr !== undefined) addr = topicData.addr;
                        if (topicData.side !== undefined) side = topicData.side;
                        if (topicData.channel !== undefined) channel = parseInt(topicData.channel);
                        if (topicData.voltage !== undefined) {
                            voltage = parseFloat(topicData.voltage);
                            voltageSource = "topic";
                        }
                        if (topicData.saveEeprom !== undefined) saveEeprom = topicData.saveEeprom;
                    } catch (e) {
                        // Si no es pot parsejar, continua amb altres opcions
                    }
                }
                
                // Prioritat 2: Si payload és un objecte JSON amb paràmetres
                if (voltage === null && msg.payload && typeof msg.payload === 'object' && !Array.isArray(msg.payload)) {
                    if (msg.payload.addr !== undefined) addr = msg.payload.addr;
                    if (msg.payload.side !== undefined) side = msg.payload.side;
                    if (msg.payload.channel !== undefined) channel = parseInt(msg.payload.channel);
                    if (msg.payload.voltage !== undefined) {
                        voltage = parseFloat(msg.payload.voltage);
                        voltageSource = "payload-object";
                    }
                    if (msg.payload.saveEeprom !== undefined) saveEeprom = msg.payload.saveEeprom;
                }
                
                // Prioritat 3: Si payload és un número O un string que representa un número
                if (voltage === null) {
                    // Intentar convertir a número (funciona amb strings i números)
                    const parsedVoltage = parseFloat(msg.payload);
                    
                    // Comprovar que és un número vàlid
                    if (!isNaN(parsedVoltage) && isFinite(parsedVoltage)) {
                        voltage = parsedVoltage;
                        voltageSource = typeof msg.payload === 'string' ? "payload-string" : "payload-number";
                    }
                }
                
                // Prioritat 4: Si no s'ha pogut determinar el voltatge, utilitzem el valor per defecte del node
                if (voltage === null) {
                    voltage = defaultVoltage;
                    voltageSource = "config";
                }
                
                // Debug: Registrar d'on ve el voltatge
                node.log(`Voltage source: ${voltageSource}, value: ${voltage}V (input type: ${typeof msg.payload})`);
                
                // Validació del voltatge
                if (typeof voltage !== 'number' || isNaN(voltage) || !isFinite(voltage) || voltage < 0 || voltage > 10) {
                    node.error(`El voltatge ha de ser un nombre entre 0 i 10 volts (rebut: ${voltage}, tipus: ${typeof voltage}, input: ${msg.payload})`);
                    node.status({fill: "red", shape: "dot", text: "error: voltatge invàlid"});
                    return;
                }
                
                // Validació del canal
                if (channel < 1 || channel > 4) {
                    node.error('El canal ha de ser entre 1 i 4');
                    node.status({fill: "red", shape: "dot", text: "error: canal invàlid"});
                    return;
                }

                node.status({fill: "blue", shape: "dot", text: `escrivint ${voltage.toFixed(2)}V...`});

                // Convertir voltatge a valor DAC (0-4095)
                let dacValue = v2aout(voltage);
                
                // Si cal desar a l'EEPROM, sumar 4096 al valor DAC
                if (saveEeprom) {
                    dacValue += 4096;
                }

                // Crida a la funció aout
                await aout(addr, side, channel, dacValue);

                // Calcular el valor DAC real (sense l'offset de l'EEPROM)
                const realDacValue = saveEeprom ? dacValue - 4096 : dacValue;

                // Prepara la sortida
                msg.payload = {
                    addr: addr,
                    side: side,
                    channel: channel,
                    voltage: voltage,
                    voltageSource: voltageSource, // Indica d'on ve el voltatge
                    dacValue: realDacValue,
                    dacHex: `0x${realDacValue.toString(16).toUpperCase().padStart(4, '0')}`,
                    savedToEeprom: saveEeprom,
                    success: true
                };

                const statusText = saveEeprom ? 
                    `${voltage.toFixed(2)}V (EEPROM)` : 
                    `${voltage.toFixed(2)}V`;
                node.status({fill: "green", shape: "dot", text: statusText});
                node.send(msg);

            } catch (error) {
                node.error(`Error en aout: ${error.message}`);
                node.status({fill: "red", shape: "ring", text: "error"});
                msg.payload = { error: error.message };
                node.send(msg);
            }
        });
    }

    RED.nodes.registerType("aout", AoutNode);
}
