/**
 * Node-RED node: ain
 * Llegeix una entrada analògica (-10V a +10V) de l'ADC
 */

module.exports = function(RED) {
    const { ain, ain2v } = require('./i2c-iotv4');

    function AinNode(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        
        // Configuració del node
        const defaultAddr = config.addr || "0000";
        const defaultSide = config.side || "A";
        const defaultChannel = parseInt(config.channel) || 1;

        node.on('input', async function(msg) {
            try {
                // Determinar addr, side i channel
                let addr = defaultAddr;
                let side = defaultSide;
                let channel = defaultChannel;
                
                // Prioritat 1: msg.topic amb JSON té prioritat sobre tot
                if (msg.topic && typeof msg.topic === 'string') {
                    try {
                        const topicData = JSON.parse(msg.topic);
                        if (topicData.addr !== undefined) addr = topicData.addr;
                        if (topicData.side !== undefined) side = topicData.side;
                        if (topicData.channel !== undefined) channel = parseInt(topicData.channel);
                    } catch (e) {
                        // Si no es pot parsejar, continua amb altres opcions
                    }
                }
                
                // Prioritat 2: Si payload és un objecte amb els paràmetres
                if (msg.payload && typeof msg.payload === 'object' && !Array.isArray(msg.payload)) {
                    if (msg.payload.addr !== undefined) addr = msg.payload.addr;
                    if (msg.payload.side !== undefined) side = msg.payload.side;
                    if (msg.payload.channel !== undefined) channel = parseInt(msg.payload.channel);
                }
                
                // Validació del canal
                if (channel < 1 || channel > 4) {
                    node.error('El canal ha de ser entre 1 i 4');
                    node.status({fill: "red", shape: "dot", text: "error: canal invàlid"});
                    return;
                }

                node.status({fill: "blue", shape: "dot", text: "llegint..."});

                // Delay de 5ms per estabilitzar la lectura analògica i evitar valors espuris
                await new Promise(resolve => setTimeout(resolve, 5));

                // Crida a la funció ain per obtenir el valor ADC raw
                const adcValue = await ain(addr, side, channel);
                
                // Convertir el valor ADC a voltatge
                const voltage = ain2v(adcValue);

                // Sortida 1: només el voltatge
                let msg1 = { ...msg };
                msg1.payload = voltage;

                // Sortida 2: informació completa (estil din)
                let msg2 = { ...msg };
                msg2.payload = {
                    addr: addr,
                    side: side,
                    channel: channel,
                    voltage: voltage,
                    adcValue: adcValue,
                    adcHex: `0x${adcValue.toString(16).toUpperCase().padStart(4, '0')}`,
                    adcBinary: adcValue.toString(2).padStart(16, '0')
                };

                node.status({fill: "green", shape: "dot", text: `${voltage}V`});
                
                // Enviar a les dues sortides
                node.send([msg1, msg2]);

            } catch (error) {
                node.error(`Error en ain: ${error.message}`);
                node.status({fill: "red", shape: "ring", text: "error"});
                
                let errorMsg1 = { ...msg, payload: null };
                let errorMsg2 = { ...msg, payload: { error: error.message } };
                node.send([errorMsg1, errorMsg2]);
            }
        });
    }

    RED.nodes.registerType("ain", AinNode);
}
